<?php
require_once dirname(__DIR__) . 
'/config/config.php';

// Check if the user is logged in
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: login.php");
    exit;
}

$user_id = $_SESSION["id"];
$username = $_SESSION["username"];
$email = "";
$profile_bio = "";
$profile_avatar_url = "";

$bio_err = "";
$avatar_err = "";
$message = "";

// Fetch current user data
$sql_get = "SELECT email, profile_bio, profile_avatar_url FROM users WHERE id = :user_id";
if ($stmt_get = $pdo->prepare($sql_get)) {
    $stmt_get->bindParam(":user_id", $user_id, PDO::PARAM_INT);
    if ($stmt_get->execute()) {
        $user_data = $stmt_get->fetch();
        if ($user_data) {
            $email = $user_data["email"];
            $profile_bio = $user_data["profile_bio"];
            $profile_avatar_url = $user_data["profile_avatar_url"];
        }
    }
    unset($stmt_get);
}

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $new_bio = trim($_POST["profile_bio"]);
    // Basic validation for bio length (optional)
    if (strlen($new_bio) > 1000) { // Example limit
        $bio_err = "Bio cannot exceed 1000 characters.";
    }

    // Handle avatar upload (basic example - needs more robust handling)
    $new_avatar_url = $profile_avatar_url; // Keep old one by default
    if (isset($_FILES["profile_avatar"]) && $_FILES["profile_avatar"]["error"] == 0) {
        $target_dir = "uploads/avatars/";
        // Ensure upload directory exists and is writable
        if (!is_dir($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        $file_extension = strtolower(pathinfo($_FILES["profile_avatar"]["name"], PATHINFO_EXTENSION));
        $safe_filename = "user_" . $user_id . "_" . time() . "." . $file_extension;
        $target_file = $target_dir . $safe_filename;
        $allowed_types = ["jpg", "jpeg", "png", "gif"];
        $max_size = 2 * 1024 * 1024; // 2MB

        // Check file type
        if (!in_array($file_extension, $allowed_types)) {
            $avatar_err = "Sorry, only JPG, JPEG, PNG & GIF files are allowed.";
        } 
        // Check file size
        elseif ($_FILES["profile_avatar"]["size"] > $max_size) {
            $avatar_err = "Sorry, your file is too large (Max 2MB).";
        } 
        // Attempt to move uploaded file
        elseif (!move_uploaded_file($_FILES["profile_avatar"]["tmp_name"], $target_file)) {
            $avatar_err = "Sorry, there was an error uploading your file.";
        } else {
            // Successfully uploaded, update the URL
            $new_avatar_url = $target_file; 
            // Optionally, delete the old avatar file if it exists and is different
            if (!empty($profile_avatar_url) && $profile_avatar_url != $new_avatar_url && file_exists($profile_avatar_url)) {
                 unlink($profile_avatar_url);
            }
        }
    }

    // If no errors, update the database
    if (empty($bio_err) && empty($avatar_err)) {
        $sql_update = "UPDATE users SET profile_bio = :profile_bio, profile_avatar_url = :profile_avatar_url WHERE id = :user_id";
        if ($stmt_update = $pdo->prepare($sql_update)) {
            $stmt_update->bindParam(":profile_bio", $new_bio, PDO::PARAM_STR);
            $stmt_update->bindParam(":profile_avatar_url", $new_avatar_url, PDO::PARAM_STR);
            $stmt_update->bindParam(":user_id", $user_id, PDO::PARAM_INT);

            if ($stmt_update->execute()) {
                $message = "Profile updated successfully.";
                // Update local variables for display
                $profile_bio = $new_bio;
                $profile_avatar_url = $new_avatar_url;
            } else {
                $message = "Error updating profile.";
            }
            unset($stmt_update);
        }
    }
}

unset($pdo);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Edit Profile - Cortlak.com</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: sans-serif; background-color: #f0f0f0; margin: 0; padding: 20px; }
        .container { max-width: 600px; margin: 20px auto; background-color: #fff; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h2 { text-align: center; color: #333; margin-bottom: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        input[type="text"], input[type="email"], textarea, input[type="file"] { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        input[readonly] { background-color: #e9ecef; cursor: not-allowed; }
        textarea { resize: vertical; min-height: 100px; }
        .help-block { color: #dc3545; font-size: 0.9em; margin-top: 5px; }
        .message { padding: 10px; margin-bottom: 15px; border-radius: 4px; }
        .message.success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .message.error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .btn { display: inline-block; padding: 10px 20px; background-color: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; font-size: 1em; text-align: center; text-decoration: none; }
        .btn:hover { background-color: #0056b3; }
        .avatar-preview { width: 100px; height: 100px; border-radius: 50%; margin-bottom: 10px; background-color: #ccc; background-size: cover; background-position: center; }
        .flex-container { display: flex; justify-content: space-between; align-items: center; }
    </style>
</head>
<body>

<div class="container">
    <h2>Edit Profile</h2>

    <?php if (!empty($message)): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post" enctype="multipart/form-data">
        <div class="form-group">
            <label>Username</label>
            <input type="text" value="<?php echo htmlspecialchars($username); ?>" readonly>
        </div>

        <div class="form-group">
            <label>Email</label>
            <input type="email" value="<?php echo htmlspecialchars($email); ?>" readonly>
        </div>

        <div class="form-group <?php echo (!empty($bio_err)) ? 'has-error' : ''; ?>">
            <label for="profile_bio">About Me (Bio)</label>
            <textarea id="profile_bio" name="profile_bio" placeholder="Tell us something about yourself..."><?php echo htmlspecialchars($profile_bio); ?></textarea>
            <span class="help-block"><?php echo $bio_err; ?></span>
        </div>

        <div class="form-group <?php echo (!empty($avatar_err)) ? 'has-error' : ''; ?>">
            <label for="profile_avatar">Profile Avatar</label>
            <div class="avatar-preview" style="background-image: url('<?php echo htmlspecialchars($profile_avatar_url ?: 'placeholder_avatar.png'); ?>');"></div>
            <input type="file" id="profile_avatar" name="profile_avatar" accept="image/png, image/jpeg, image/gif">
            <span class="help-block"><?php echo $avatar_err; ?></span>
        </div>

        <div class="form-group flex-container">
            <input type="submit" class="btn" value="Save Changes">
            <a href="profile.php" style="color: #6c757d;">Back to Profile</a>
        </div>
    </form>
</div>

</body>
</html>
