<?php
require_once dirname(__DIR__) . '/config/config.php';

// Check if the user is logged in, otherwise redirect to login page
if(!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true){
    header("location: login.php");
    exit;
}

// Fetch groups the user is a member of
$groups = [];
$user_id = $_SESSION["id"];

$sql = "SELECT g.id, g.name 
        FROM groups g
        JOIN group_members gm ON g.id = gm.group_id
        WHERE gm.user_id = :user_id
        ORDER BY g.name";

if($stmt = $pdo->prepare($sql)){
    $stmt->bindParam(":user_id", $user_id, PDO::PARAM_INT);
    if($stmt->execute()){
        $groups = $stmt->fetchAll();
    }
    unset($stmt);
}

// Placeholder for selected group and channels - will be dynamic later
$selected_group_id = isset($_GET['group_id']) ? (int)$_GET['group_id'] : null;
$channels = [];
$selected_channel_id = isset($_GET['channel_id']) ? (int)$_GET['channel_id'] : null;
$messages = [];
$current_group_name = "Select a Group";
$current_channel_name = "Select a Channel";

if ($selected_group_id) {
    // Verify user is a member of the selected group
    $sql_check_member = "SELECT 1 FROM group_members WHERE user_id = :user_id AND group_id = :group_id";
    if ($stmt_check = $pdo->prepare($sql_check_member)) {
        $stmt_check->bindParam(":user_id", $user_id, PDO::PARAM_INT);
        $stmt_check->bindParam(":group_id", $selected_group_id, PDO::PARAM_INT);
        $stmt_check->execute();
        if ($stmt_check->rowCount() > 0) {
            // Fetch group name
            $sql_group_name = "SELECT name FROM groups WHERE id = :group_id";
            if($stmt_group_name = $pdo->prepare($sql_group_name)) {
                $stmt_group_name->bindParam(":group_id", $selected_group_id, PDO::PARAM_INT);
                if($stmt_group_name->execute()){
                    $group_info = $stmt_group_name->fetch();
                    if($group_info) {
                        $current_group_name = $group_info['name'];
                    }
                }
                unset($stmt_group_name);
            }

            // Fetch channels for the selected group
            $sql_channels = "SELECT id, name, type FROM channels WHERE group_id = :group_id ORDER BY type, name";
            if($stmt_channels = $pdo->prepare($sql_channels)){
                $stmt_channels->bindParam(":group_id", $selected_group_id, PDO::PARAM_INT);
                if($stmt_channels->execute()){
                    $channels = $stmt_channels->fetchAll();
                }
                unset($stmt_channels);
            }

            // If a channel is selected, fetch its name and messages
            if ($selected_channel_id) {
                 // Verify channel belongs to the selected group
                 $sql_check_channel = "SELECT name, type FROM channels WHERE id = :channel_id AND group_id = :group_id";
                 if ($stmt_check_channel = $pdo->prepare($sql_check_channel)) {
                     $stmt_check_channel->bindParam(":channel_id", $selected_channel_id, PDO::PARAM_INT);
                     $stmt_check_channel->bindParam(":group_id", $selected_group_id, PDO::PARAM_INT);
                     $stmt_check_channel->execute();
                     $channel_info = $stmt_check_channel->fetch();
                     if ($channel_info) {
                         $current_channel_name = ($channel_info['type'] == 'text' ? '#' : '🔊 ') . $channel_info['name'];
                         // Fetch messages for the selected text channel
                         if ($channel_info['type'] == 'text') {
                             $sql_messages = "SELECT m.id, m.content, m.created_at, u.username 
                                             FROM messages m 
                                             JOIN users u ON m.user_id = u.id 
                                             WHERE m.channel_id = :channel_id 
                                             ORDER BY m.created_at ASC LIMIT 100"; // Limit messages for now
                             if($stmt_messages = $pdo->prepare($sql_messages)){
                                 $stmt_messages->bindParam(":channel_id", $selected_channel_id, PDO::PARAM_INT);
                                 if($stmt_messages->execute()){
                                     $messages = $stmt_messages->fetchAll();
                                 }
                                 unset($stmt_messages);
                             }
                         }
                     } else {
                         // Invalid channel for this group, reset
                         $selected_channel_id = null;
                     }
                     unset($stmt_check_channel);
                 }
            }

        } else {
            // User is not a member of the selected group, reset
            $selected_group_id = null;
            $current_group_name = "Select a Group";
        }
        unset($stmt_check);
    }
}


// Close connection
unset($pdo);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Cortlak.com</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { margin: 0; font-family: sans-serif; background-color: #36393f; color: #dcddde; display: flex; height: 100vh; overflow: hidden; }
        .sidebar { width: 70px; background-color: #202225; display: flex; flex-direction: column; align-items: center; padding-top: 12px; flex-shrink: 0; }
        .group-icon { width: 50px; height: 50px; background-color: #36393f; border-radius: 50%; margin-bottom: 8px; display: flex; align-items: center; justify-content: center; font-size: 10px; text-align: center; cursor: pointer; transition: border-radius 0.3s ease, background-color 0.3s ease; overflow: hidden; text-decoration: none; color: #dcddde; }
        .group-icon:hover, .group-icon.active { border-radius: 15px; background-color: #7289da; }
        .group-icon.add { background-color: #43b581; font-size: 24px; }
        .group-icon.add:hover { background-color: #3ca574; }

        .channels-panel { width: 240px; background-color: #2f3136; display: flex; flex-direction: column; flex-shrink: 0; }
        .group-header { padding: 12px 16px; border-bottom: 1px solid rgba(0,0,0,0.2); font-weight: bold; color: #fff; height: 48px; box-sizing: border-box; display: flex; align-items: center; }
        .channels-list { flex-grow: 1; overflow-y: auto; padding: 10px 8px; }
        .channel-category { color: #8e9297; font-size: 12px; font-weight: bold; text-transform: uppercase; margin-bottom: 5px; padding: 6px 8px; }
        .channel-item { display: flex; align-items: center; padding: 6px 8px; border-radius: 3px; margin-bottom: 2px; cursor: pointer; color: #8e9297; }
        .channel-item:hover { background-color: rgba(79,84,92,0.16); color: #dcddde; }
        .channel-item.active { background-color: rgba(79,84,92,0.32); color: #fff; }
        .channel-icon { margin-right: 6px; font-weight: bold; }
        .user-panel { background-color: #292b2f; padding: 10px; display: flex; align-items: center; border-top: 1px solid rgba(0,0,0,0.2); }
        .user-avatar { width: 32px; height: 32px; background-color: #7289da; border-radius: 50%; margin-right: 8px; }
        .user-info span { display: block; font-size: 14px; }
        .user-info .username { font-weight: bold; color: #fff; }
        .user-info .usertag { font-size: 11px; color: #b9bbbe; }
        .user-actions { margin-left: auto; display: flex; gap: 5px; }
        .user-actions button { background: none; border: none; color: #b9bbbe; cursor: pointer; font-size: 16px; padding: 4px; }
        .user-actions button:hover { color: #fff; }

        .main-content { flex-grow: 1; display: flex; flex-direction: column; background-color: #36393f; }
        .chat-header { height: 48px; border-bottom: 1px solid rgba(0,0,0,0.2); padding: 0 16px; display: flex; align-items: center; font-weight: 600; color: #fff; flex-shrink: 0; }
        .chat-messages { flex-grow: 1; overflow-y: auto; padding: 16px; display: flex; flex-direction: column-reverse; /* Newest messages at the bottom */ }
        .message { display: flex; margin-bottom: 16px; }
        .message-avatar { width: 40px; height: 40px; border-radius: 50%; background-color: #7289da; margin-right: 16px; flex-shrink: 0; }
        .message-content { display: flex; flex-direction: column; }
        .message-header { display: flex; align-items: baseline; margin-bottom: 4px; }
        .message-username { font-weight: 500; color: #fff; margin-right: 8px; font-size: 1em; }
        .message-timestamp { color: #72767d; font-size: 0.75em; }
        .message-body { color: #dcddde; font-size: 0.95em; line-height: 1.4; }
        .chat-input { display: flex; padding: 16px; border-top: 1px solid rgba(0,0,0,0.2); background-color: #40444b; }
        .chat-input input[type="text"] { flex-grow: 1; background-color: #40444b; border: none; color: #dcddde; padding: 10px; border-radius: 5px; outline: none; }
        .chat-input input[type="text"]::placeholder { color: #72767d; }

        /* Simple scrollbar styling */
        ::-webkit-scrollbar { width: 8px; }
        ::-webkit-scrollbar-track { background: #2e3338; }
        ::-webkit-scrollbar-thumb { background: #202225; border-radius: 4px; }
    </style>
</head>
<body>

    <div class="sidebar">
        <a href="#" class="group-icon" title="Home">DM</a> <!-- Placeholder for DMs -->
        <?php foreach ($groups as $group): ?>
            <a href="?group_id=<?php echo $group['id']; ?>" 
               class="group-icon <?php echo ($selected_group_id == $group['id']) ? 'active' : ''; ?>" 
               title="<?php echo htmlspecialchars($group['name']); ?>">
               <?php echo htmlspecialchars(strtoupper(substr($group['name'], 0, 1))); // Simple initial as icon ?>
            </a>
        <?php endforeach; ?>
        <a href="create_group.php" class="group-icon add" title="Add a Server">+</a>
    </div>

    <div class="channels-panel">
        <div class="group-header">
            <?php echo htmlspecialchars($current_group_name); ?>
        </div>
        <div class="channels-list">
            <?php if ($selected_group_id): ?>
                <div class="channel-category">Text Channels</div>
                <?php foreach ($channels as $channel): ?>
                    <?php if ($channel['type'] == 'text'): ?>
                        <a href="?group_id=<?php echo $selected_group_id; ?>&channel_id=<?php echo $channel['id']; ?>" 
                           class="channel-item <?php echo ($selected_channel_id == $channel['id']) ? 'active' : ''; ?>">
                            <span class="channel-icon">#</span> <?php echo htmlspecialchars($channel['name']); ?>
                        </a>
                    <?php endif; ?>
                <?php endforeach; ?>

                <div class="channel-category">Voice Channels</div>
                 <?php foreach ($channels as $channel): ?>
                    <?php if ($channel['type'] == 'voice'): ?>
                        <a href="#" class="channel-item"> <!-- Voice channel link TBD -->
                            <span class="channel-icon">🔊</span> <?php echo htmlspecialchars($channel['name']); ?>
                        </a>
                    <?php endif; ?>
                <?php endforeach; ?>
            <?php else: ?>
                 <div style="padding: 10px; color: #8e9297;">Select a group from the left.</div>
            <?php endif; ?>
        </div>
        <div class="user-panel">
            <div class="user-avatar"></div>
            <div class="user-info">
                <span class="username"><?php echo htmlspecialchars($_SESSION["username"]); ?></span>
                <span class="usertag">#0001</span> <!-- Placeholder tag -->
            </div>
            <div class="user-actions">
                <button title="Mute">🎤</button>
                <button title="Deafen">🎧</button>
                <a href="logout.php" title="Logout"><button>⚙️</button></a>
            </div>
        </div>
    </div>

    <div class="main-content">
        <div class="chat-header">
            <?php echo htmlspecialchars($current_channel_name); ?>
        </div>
        <div class="chat-messages">
            <?php if ($selected_channel_id && !empty($messages)): ?>
                <?php foreach (array_reverse($messages) as $message): // Display newest first if needed, or adjust query ?>
                    <div class="message">
                        <div class="message-avatar"></div>
                        <div class="message-content">
                            <div class="message-header">
                                <span class="message-username"><?php echo htmlspecialchars($message['username']); ?></span>
                                <span class="message-timestamp"><?php echo date("h:i A", strtotime($message['created_at'])); ?></span>
                            </div>
                            <div class="message-body">
                                <?php echo nl2br(htmlspecialchars($message['content'])); ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php elseif ($selected_channel_id): ?>
                 <div style="text-align: center; color: #8e9297; margin-top: 20px;">No messages in this channel yet.</div>
            <?php else: ?>
                <div style="text-align: center; color: #8e9297; margin-top: 20px;">Select a channel to start chatting.</div>
            <?php endif; ?>
        </div>
        <div class="chat-input">
            <?php if ($selected_channel_id): ?>
            <form action="send_message.php" method="post" style="display: flex; width: 100%;">
                <input type="hidden" name="channel_id" value="<?php echo $selected_channel_id; ?>">
                <input type="text" name="message" placeholder="Message <?php echo htmlspecialchars($current_channel_name); ?>" autocomplete="off" required>
                <button type="submit" style="margin-left: 10px; padding: 10px 15px; background-color: #7289da; color: white; border: none; border-radius: 5px; cursor: pointer;">Send</button>
            </form>
            <?php else: ?>
             <input type="text" placeholder="Select a channel to send messages" disabled>
            <?php endif; ?>
        </div>
    </div>

</body>
</html>

