// Cortlak.com - voice.js

let localStream = null;
let peerConnections = {}; // Store peer connections, keyed by user ID
let currentVoiceChannelId = null;

// Placeholder for signaling functions (to be implemented)
function sendSignal(toUserId, data) {
    console.log(`Sending signal to ${toUserId}:`, data);
    // This will eventually use WebSockets or another mechanism
    // For now, maybe use AJAX to a PHP endpoint
    fetch('app/signaling/signal.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ to_user_id: toUserId, data: data, from_user_id: myUserId, channel_id: currentVoiceChannelId })
    });
}

// Function to handle received signals (called by polling or WebSocket)
function handleSignal(fromUserId, data) {
    console.log(`Received signal from ${fromUserId}:`, data);
    let pc = peerConnections[fromUserId];

    if (data.sdp) {
        if (data.sdp.type === 'offer' && !pc) {
            // Received an offer from a new peer
            pc = createPeerConnection(fromUserId);
            peerConnections[fromUserId] = pc;
        }
        if (pc) {
            pc.setRemoteDescription(new RTCSessionDescription(data.sdp))
                .then(() => {
                    if (data.sdp.type === 'offer') {
                        return pc.createAnswer();
                    }
                })
                .then(answer => {
                    if (answer) {
                        return pc.setLocalDescription(answer);
                    }
                })
                .then(() => {
                    if (pc.localDescription && pc.localDescription.type === 'answer') {
                        sendSignal(fromUserId, { sdp: pc.localDescription });
                    }
                })
                .catch(e => console.error("Error handling SDP:", e));
        }
    } else if (data.ice && pc) {
        pc.addIceCandidate(new RTCIceCandidate(data.ice))
            .catch(e => console.error("Error adding ICE candidate:", e));
    }
}

// Function to create a peer connection
function createPeerConnection(peerUserId) {
    const pc = new RTCPeerConnection({ iceServers: [{ urls: 'stun:stun.l.google.com:19302' }] });

    pc.onicecandidate = event => {
        if (event.candidate) {
            sendSignal(peerUserId, { ice: event.candidate });
        }
    };

    pc.ontrack = event => {
        console.log(`Track received from ${peerUserId}`);
        const audio = document.createElement('audio');
        audio.autoplay = true;
        audio.srcObject = event.streams[0];
        audio.id = `audio-${peerUserId}`;
        document.body.appendChild(audio); // Append somewhere appropriate
    };

    // Add local stream tracks
    if (localStream) {
        localStream.getTracks().forEach(track => pc.addTrack(track, localStream));
    }

    return pc;
}

// Function to join a voice channel
async function joinVoiceChannel(channelId) {
    if (currentVoiceChannelId === channelId) return; // Already in this channel

    console.log(`Joining voice channel ${channelId}`);
    leaveVoiceChannel(); // Leave current channel first

    currentVoiceChannelId = channelId;

    try {
        localStream = await navigator.mediaDevices.getUserMedia({ audio: true, video: false });
        // Update UI to show connected status, mute/deafen controls etc.
        updateVoiceStatusUI(true);

        // Notify server that we joined (e.g., via AJAX)
        // This would ideally trigger signaling to other users in the channel
        fetch('app/voice/join.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ channel_id: channelId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && data.users) {
                console.log('Users in channel:', data.users);
                // Initiate connections to other users already in the channel
                data.users.forEach(userId => {
                    if (userId !== myUserId) { // myUserId needs to be defined globally
                        const pc = createPeerConnection(userId);
                        peerConnections[userId] = pc;
                        pc.createOffer()
                            .then(offer => pc.setLocalDescription(offer))
                            .then(() => {
                                sendSignal(userId, { sdp: pc.localDescription });
                            })
                            .catch(e => console.error("Error creating offer:", e));
                    }
                });
            }
        });

        // Start polling for signals (simple implementation)
        startSignalingPolling();

    } catch (err) {
        console.error('Error accessing microphone:', err);
        alert('Could not access microphone. Please check permissions.');
        currentVoiceChannelId = null;
    }
}

// Function to leave a voice channel
function leaveVoiceChannel() {
    if (!currentVoiceChannelId) return;

    console.log(`Leaving voice channel ${currentVoiceChannelId}`);

    // Notify server
    fetch('app/voice/leave.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ channel_id: currentVoiceChannelId })
    });

    // Close all peer connections
    for (const userId in peerConnections) {
        peerConnections[userId].close();
        const audioElement = document.getElementById(`audio-${userId}`);
        if (audioElement) audioElement.remove();
    }
    peerConnections = {};

    // Stop local media stream
    if (localStream) {
        localStream.getTracks().forEach(track => track.stop());
        localStream = null;
    }

    // Stop polling
    stopSignalingPolling();

    // Update UI
    updateVoiceStatusUI(false);
    currentVoiceChannelId = null;
}

// --- Signaling Polling (Basic Example) ---
let pollingInterval = null;

function startSignalingPolling() {
    stopSignalingPolling(); // Clear any existing interval
    pollingInterval = setInterval(() => {
        if (!currentVoiceChannelId) return;
        fetch(`app/signaling/poll.php?channel_id=${currentVoiceChannelId}&user_id=${myUserId}`)
            .then(response => response.json())
            .then(signals => {
                if (signals && signals.length > 0) {
                    signals.forEach(signal => {
                        handleSignal(signal.from_user_id, signal.data);
                    });
                }
            })
            .catch(e => console.error("Polling error:", e));
    }, 3000); // Poll every 3 seconds (adjust as needed)
}

function stopSignalingPolling() {
    if (pollingInterval) {
        clearInterval(pollingInterval);
        pollingInterval = null;
    }
}

// --- UI Updates ---
function updateVoiceStatusUI(isConnected) {
    const statusElement = document.getElementById('voice-status'); // Assume an element exists
    if (statusElement) {
        statusElement.textContent = isConnected ? `Connected to channel ${currentVoiceChannelId}` : 'Not connected';
    }
    // Add more UI updates (e.g., enable/disable mute/deafen buttons)
}

// --- Initialization ---
// Need to get the current user's ID (e.g., from a PHP variable embedded in the HTML)
// const myUserId = <?php echo json_encode($_SESSION['id']); ?>; // Example
let myUserId = null; // Needs to be set

document.addEventListener('DOMContentLoaded', () => {
    // Try to get user ID if embedded in HTML
    const userIdElement = document.getElementById('user-id-data');
    if (userIdElement) {
        myUserId = parseInt(userIdElement.getAttribute('data-user-id'), 10);
    }

    // Add event listeners to voice channel links
    document.querySelectorAll('.channel-item[data-channel-type="voice"]').forEach(link => {
        link.addEventListener('click', (event) => {
            event.preventDefault();
            const channelId = parseInt(link.getAttribute('data-channel-id'), 10);
            if (channelId) {
                joinVoiceChannel(channelId);
            }
        });
    });

    // Add listener for a potential disconnect button
    const disconnectButton = document.getElementById('disconnect-voice');
    if (disconnectButton) {
        disconnectButton.addEventListener('click', leaveVoiceChannel);
    }
});

// Add placeholder for user list in voice channel
function updateUserListInVoiceChannel(users) {
    const userListElement = document.getElementById('voice-channel-users'); // Assume an element exists
    if (!userListElement) return;

    userListElement.innerHTML = ''; // Clear current list
    if (users && users.length > 0) {
        users.forEach(user => {
            const li = document.createElement('li');
            li.textContent = user.username; // Assuming user object has username
            userListElement.appendChild(li);
        });
    } else {
        userListElement.innerHTML = '<li>No one else here yet.</li>';
    }
}

