<?php
require_once dirname(__DIR__) . '/config/config.php';

// Get user ID from query string, default to logged-in user if not specified or invalid
$profile_user_id = null;
$is_own_profile = false;

if (isset($_GET['id']) && is_numeric($_GET['id'])) {
    $profile_user_id = (int)$_GET['id'];
} elseif (isset($_SESSION["loggedin"]) && $_SESSION["loggedin"] === true) {
    $profile_user_id = $_SESSION["id"];
    $is_own_profile = true;
}

if ($profile_user_id === null) {
    // If not logged in and no ID specified, redirect to login
    header("location: login.php");
    exit;
}

// Fetch user data
$user_info = null;
$sql_user = "SELECT id, username, email, profile_bio, profile_avatar_url, created_at FROM users WHERE id = :id";
if ($stmt_user = $pdo->prepare($sql_user)) {
    $stmt_user->bindParam(":id", $profile_user_id, PDO::PARAM_INT);
    if ($stmt_user->execute()) {
        $user_info = $stmt_user->fetch();
    }
    unset($stmt_user);
}

if (!$user_info) {
    // User not found
    die("User not found."); // Or show a proper 404 page
}

// Fetch stream info if the user has set it up
$stream_info = null;
$sql_stream = "SELECT title, description, is_live FROM streams WHERE user_id = :user_id";
if ($stmt_stream = $pdo->prepare($sql_stream)) {
    $stmt_stream->bindParam(":user_id", $profile_user_id, PDO::PARAM_INT);
    if ($stmt_stream->execute()) {
        $stream_info = $stmt_stream->fetch();
    }
    unset($stmt_stream);
}

// Fetch mutual groups (if viewing someone else's profile and logged in)
$mutual_groups = [];
if (!$is_own_profile && isset($_SESSION["loggedin"]) && $_SESSION["loggedin"] === true) {
    $current_user_id = $_SESSION["id"];
    $sql_mutual = "SELECT g.id, g.name 
                   FROM groups g 
                   JOIN group_members gm1 ON g.id = gm1.group_id
                   JOIN group_members gm2 ON g.id = gm2.group_id
                   WHERE gm1.user_id = :current_user_id AND gm2.user_id = :profile_user_id";
    if ($stmt_mutual = $pdo->prepare($sql_mutual)) {
        $stmt_mutual->bindParam(":current_user_id", $current_user_id, PDO::PARAM_INT);
        $stmt_mutual->bindParam(":profile_user_id", $profile_user_id, PDO::PARAM_INT);
        if ($stmt_mutual->execute()) {
            $mutual_groups = $stmt_mutual->fetchAll();
        }
        unset($stmt_mutual);
    }
}


unset($pdo);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title><?php echo htmlspecialchars($user_info['username']); ?>'s Profile - Cortlak.com</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body { font-family: sans-serif; background-color: #f0f0f0; margin: 0; padding: 20px; }
        .profile-container { max-width: 800px; margin: 20px auto; background-color: #fff; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); overflow: hidden; }
        .profile-header { background-color: #7289da; color: white; padding: 40px 20px; text-align: center; position: relative; }
        .profile-avatar { width: 120px; height: 120px; border-radius: 50%; border: 4px solid white; margin: -60px auto 10px auto; display: block; background-color: #ccc; /* Placeholder */ }
        .profile-username { font-size: 2em; font-weight: bold; margin: 0; }
        .profile-since { font-size: 0.9em; color: #eee; }
        .profile-actions { position: absolute; top: 15px; right: 15px; }
        .profile-actions a, .profile-actions button { background-color: rgba(0,0,0,0.2); color: white; border: none; padding: 8px 12px; border-radius: 5px; text-decoration: none; font-size: 0.9em; cursor: pointer; margin-left: 5px; }
        .profile-actions a:hover, .profile-actions button:hover { background-color: rgba(0,0,0,0.4); }
        .profile-body { padding: 20px; }
        .profile-section { margin-bottom: 20px; padding-bottom: 20px; border-bottom: 1px solid #eee; }
        .profile-section:last-child { border-bottom: none; margin-bottom: 0; padding-bottom: 0; }
        .profile-section h3 { margin-top: 0; color: #555; }
        .profile-bio { white-space: pre-wrap; word-wrap: break-word; color: #333; }
        .no-bio { color: #888; font-style: italic; }
        .stream-status { font-weight: bold; }
        .stream-status.live { color: red; }
        .stream-status.offline { color: gray; }
        .group-list li { margin-bottom: 5px; }
        .group-list a { text-decoration: none; color: #007bff; }
        .group-list a:hover { text-decoration: underline; }
    </style>
</head>
<body>

<div class="profile-container">
    <div class="profile-header">
        <?php if ($is_own_profile): ?>
            <div class="profile-actions">
                <a href="edit_profile.php">Edit Profile</a>
                <a href="stream_settings.php">Stream Settings</a>
                <a href="logout.php">Logout</a>
            </div>
        <?php elseif (isset($_SESSION["loggedin"]) && $_SESSION["loggedin"] === true): ?>
             <div class="profile-actions">
                <!-- Add Friend / Message buttons could go here -->
            </div>
        <?php endif; ?>
        <img src="<?php echo htmlspecialchars($user_info['profile_avatar_url'] ?: 'placeholder_avatar.png'); ?>" alt="<?php echo htmlspecialchars($user_info['username']); ?>'s Avatar" class="profile-avatar">
        <h2 class="profile-username"><?php echo htmlspecialchars($user_info['username']); ?></h2>
        <p class="profile-since">Member since <?php echo date("F j, Y", strtotime($user_info['created_at'])); ?></p>
    </div>

    <div class="profile-body">
        <div class="profile-section">
            <h3>About Me</h3>
            <?php if (!empty($user_info['profile_bio'])): ?>
                <p class="profile-bio"><?php echo nl2br(htmlspecialchars($user_info['profile_bio'])); ?></p>
            <?php else: ?>
                <p class="no-bio">This user hasn't written a bio yet.</p>
            <?php endif; ?>
        </div>

        <?php if ($stream_info): ?>
        <div class="profile-section">
            <h3>Stream</h3>
            <p>Title: <?php echo htmlspecialchars($stream_info['title']); ?></p>
            <p>Status: <span class="stream-status <?php echo $stream_info['is_live'] ? 'live' : 'offline'; ?>"><?php echo $stream_info['is_live'] ? 'LIVE' : 'Offline'; ?></span></p>
            <?php if ($stream_info['is_live']): ?>
                <p><a href="watch.php?user=<?php echo $user_info['id']; ?>">Watch Now</a></p>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <?php if (!$is_own_profile && !empty($mutual_groups)): ?>
        <div class="profile-section">
            <h3>Mutual Groups</h3>
            <ul class="group-list">
                <?php foreach ($mutual_groups as $group): ?>
                    <li><a href="index.php?group_id=<?php echo $group['id']; ?>"><?php echo htmlspecialchars($group['name']); ?></a></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php endif; ?>

        <!-- Add more sections as needed, e.g., roles, connections -->

    </div>
</div>

<p style="text-align: center;"><a href="index.php">Back to App</a></p>

</body>
</html>

